using System;
using System.Drawing;
using System.IO;
using System.Diagnostics;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;

namespace findonpaths
{
	/// <summary>
	/// A simple "Find On Paths" windows form demo example.
	///
	/// This tool can be used to find the occurences of
	/// specified executable files on the usual windows paths.
	/// It's also possible, for example to search for some *.dll
	/// file along the windows paths with this tool.
	///
	/// Example submitted by Valentino Kyriakides
	///
	/// </summary>
	public class WinForm : System.Windows.Forms.Form
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.Label label1;
		private System.Windows.Forms.TextBox searchTextBox;
		private System.Windows.Forms.CheckBox firstOccurrenceCheckBox;
		private System.Windows.Forms.Button findButton;
		private System.Windows.Forms.TextBox resultTextBox;

        /// <summary>
		/// Additionally setup variables.
		/// </summary>
		private ArrayList searchPath = new ArrayList();
		private ArrayList extensionsList = new ArrayList();
		private string[] defaultExtensions =
		{
		  ".COM", ".EXE", ".BAT", ".CMD", ".VBS",
		  ".VBE", ".JS", ".JSE", ".WSF",".WSH"
		};
		private char[] semicolon = new char[] {';'};
		private int extensionsCount = 0;
		private int directoryCount = 0;
		private bool firstMatchOnly = false;


		public WinForm()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			//
			// TODO: Add any constructor code after InitializeComponent call
			//
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose (bool disposing)
		{
			if (disposing)
			{
				if (components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose(disposing);
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.label1 = new System.Windows.Forms.Label();
			this.searchTextBox = new System.Windows.Forms.TextBox();
			this.firstOccurrenceCheckBox = new System.Windows.Forms.CheckBox();
			this.findButton = new System.Windows.Forms.Button();
			this.resultTextBox = new System.Windows.Forms.TextBox();
			this.SuspendLayout();
			// 
			// label1
			// 
			this.label1.Location = new System.Drawing.Point(8, 16);
			this.label1.Name = "label1";
			this.label1.Size = new System.Drawing.Size(248, 16);
			this.label1.TabIndex = 0;
			this.label1.Text = "Find on paths:";
			// 
			// searchTextBox
			//
			this.searchTextBox.Location = new System.Drawing.Point(8, 40);
			this.searchTextBox.Name = "searchTextBox";
			this.searchTextBox.Size = new System.Drawing.Size(488, 20);
			this.searchTextBox.TabIndex = 1;
			this.searchTextBox.Text = "bds.exe";
			// 
			// firstOccurrenceCheckBox
			// 
			this.firstOccurrenceCheckBox.Location = new System.Drawing.Point(360, 8);
			this.firstOccurrenceCheckBox.Name = "firstOccurrenceCheckBox";
			this.firstOccurrenceCheckBox.Size = new System.Drawing.Size(128, 24);
			this.firstOccurrenceCheckBox.TabIndex = 2;
			this.firstOccurrenceCheckBox.Text = "First occurrence only";
			this.firstOccurrenceCheckBox.Click += new System.EventHandler(this.firstOccurrenceCheckBox_Click);
			//
			// findButton
			// 
			this.findButton.Location = new System.Drawing.Point(504, 40);
			this.findButton.Name = "findButton";
			this.findButton.TabIndex = 3;
			this.findButton.Text = "Find";
			this.findButton.Click += new System.EventHandler(this.findButton_Click);
			// 
			// resultTextBox
			// 
			this.resultTextBox.BackColor = System.Drawing.SystemColors.Info;
			this.resultTextBox.Location = new System.Drawing.Point(8, 80);
			this.resultTextBox.Multiline = true;
			this.resultTextBox.Name = "resultTextBox";
			this.resultTextBox.ScrollBars = System.Windows.Forms.ScrollBars.Both;
			this.resultTextBox.Size = new System.Drawing.Size(568, 216);
			this.resultTextBox.TabIndex = 4;
			this.resultTextBox.Text = "";
			// 
			// WinForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(584, 301);
			this.Controls.Add(this.resultTextBox);
			this.Controls.Add(this.searchTextBox);
			this.Controls.Add(this.findButton);
			this.Controls.Add(this.firstOccurrenceCheckBox);
			this.Controls.Add(this.label1);
			this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedSingle;
			this.Name = "WinForm";
			this.Text = "Find On Paths";
			this.ResumeLayout(false);
		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main() 
		{
			Application.Run(new WinForm());
		}
		
		private void findButton_Click(object sender, System.EventArgs e)
		{
			resultTextBox.Text = "";
			PerformPathsSearch();
		}

		private void firstOccurrenceCheckBox_Click(object sender, System.EventArgs e)
		{
			if (firstOccurrenceCheckBox.Checked == true)
				firstMatchOnly = true;
			else
				firstMatchOnly = false;
		}

		/// <summary>
		///  First sets up an array of search locations.
		///
		///  The order directories are searched is:
		///  - the directory where this executable itself is located
		///  - the current directory.
		///  - the value of GetSystemDirectory()
		///  - the value of GetWindowsDirectory()
		///  - the directories along the environment path variable
		///
		///  Afterwards the find for the specified entry of the
		///  search text box is performed on the directories along
		///  these these paths.
		/// </summary>
		void PerformPathsSearch()
		{
			// Add the directory where this app itself resides to the search path list.
			Process currentProcess = System.Diagnostics.Process.GetCurrentProcess();
			ProcessModule processModul = currentProcess.MainModule;
			string actualDir = Path.GetDirectoryName(processModul.FileName);
			if (actualDir.Length > 1 && '\\' == actualDir[actualDir.Length-1])
				actualDir.Remove(actualDir.Length-1,1);
			searchPath.Add(actualDir);

			// Add current directory and system directory to the search path list
			string currentDir = Environment.CurrentDirectory;
			if (currentDir.Length > 1 && '\\' == currentDir[currentDir.Length-1])
				currentDir.Remove(currentDir.Length-1,1);
			addElementToList(currentDir, searchPath);

			string systemDir = Environment.SystemDirectory;
			if (systemDir.Length > 1 && '\\' == systemDir[systemDir.Length-1])
				systemDir.Remove(systemDir.Length-1,1);
			addElementToList(systemDir, searchPath);


			// Add the windows directory to the search path list
			string windowsDir = Directory.GetParent(systemDir).FullName;
			if (windowsDir.Length > 1 && '\\' == windowsDir[windowsDir.Length-1])
				windowsDir.Remove(windowsDir.Length-1,1);
			addElementToList(windowsDir, searchPath);

			// Add the environment path var entries to the search path list
			string pathString = Environment.GetEnvironmentVariable("Path");
			if (pathString != string.Empty)
			{
				string[] pathElements = pathString.Split(semicolon);
				int pathElementCount = pathElements.GetLength(0);
				for (int i=0; i<pathElementCount; i++)
				{
				  string theElement = pathElements[i];
				  if (theElement.Length > 1 && '\\' == theElement[theElement.Length-1])
				  	theElement.Remove(theElement.Length-1,1);
				  addElementToList(theElement, searchPath);
				}
			}
			directoryCount = searchPath.Count;

			// Build a file type extension array list based
			// on the value of the PATHEXT env variable
			string[] extensionArray = new string[] {};
			string pathExt = Environment.GetEnvironmentVariable("PathExt");
			if (string.Empty == pathExt)
				defaultExtensions.CopyTo(extensionArray,0);
			else
				extensionArray = pathExt.Split(semicolon);

			extensionsCount = extensionArray.Length;
			for (int i=0; i<extensionsCount; i++)
				addElementToList(extensionArray[i], extensionsList);

			// Take the entry from the search text box and find all
			// matching executable files that live along the search
			// paths (or just find the first occurrence, in case the
			// first occurrence check box is checked).
			if (searchTextBox.Text != string.Empty)
				FindOnPaths(searchTextBox.Text);
		}

		/// <summary>
		/// Tries to find the specified search item entry along
		/// the directories of the search path list. If the search
		/// item (file) exists on some path, the found path is printed
		/// out into the result text box.
		/// </summary>
		void FindOnPaths(string searchItem)
		{
		  // Iterate through the directories on the specified paths
		  for (int i=0; i<directoryCount; i++)
		  {
			bool fileExists = false;

			// Iterate through the file type extensions on the search paths,
			// if a corresponding file matches and exists print it out onto
			// the result text box
			string extStr = Path.GetExtension(searchItem);
			if (extStr != string.Empty)
			{
			  string theFile = string.Format("{0}\\{1}", (string)searchPath[i], searchItem);
			  if ((fileExists = File.Exists(theFile)))
				resultTextBox.AppendText(theFile);
			}
			else
			{
			  for (int j=0; j<extensionsCount; j++)
			  {
				string theFile = string.Format("{0}\\{1}{2}",
				  (string)searchPath[i], searchItem, extensionsList[j]);
				if ((fileExists = File.Exists(theFile)))
				{
				  resultTextBox.AppendText(theFile);
				  if (firstMatchOnly)
				  	break;
				}
			  }
			}
			if (fileExists && firstMatchOnly)
				break;
		  }
		}

		/// <summary>
		/// Adds an element to the given array list.
		/// </summary>
		static void addElementToList(string element, ArrayList arrayList)
		{
		  for (int i=0; i<arrayList.Count; i++)
		  { 
			if (0 == string.Compare(element, (string)arrayList[i], true))
				return;
		  }
		  arrayList.Add(element);
		}
	}
}
